package gov.va.med.mhv.phr.appointments.decorator;

import java.io.Serializable;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.phr.decorator.DecoratorUtils;
import gov.va.med.mhv.phr.transfer.Appointment;
import gov.va.med.mhv.phr.transfer.ImpreciseDateComposite;

/*
 * This class is the decorator for the Appointment Summary.
 *
 * @author - Jeff Robinson (Aquilent, Inc.)
 */
public class AppointmentDecorator implements Serializable{

    private static final long serialVersionUID = -7069372160082275849L;

    protected static final String COMPENSATION_APPT = "This is a Compensation & Pension Appointment";
    protected static final String CP = "COMPENSATION & PENSION";
    protected static final String CANCELLED = "CANCELLED";

    protected static final String BLANK = StringEscapeUtils.escapeHtml("");

    protected static String escapedValueOrBlank(String value) {
        return StringUtils.isBlank(value) ? BLANK : escapedValue(value);
    }

    protected static String escapedValue(String value) {
        return StringEscapeUtils.escapeHtml(value);
    }

    private final Appointment appointment;

    /*
     * Contructor for the AppointmentDecorator class
     *
     *  @param appointment The Appointment object to decorate
     */
    @SuppressWarnings("unchecked")
    public AppointmentDecorator(Appointment appointment) {
        Precondition.assertNotNull("appointment", appointment);
        this.appointment = appointment;
    }

    /*
     * This method retrieves and formats the Appointment Date/Time for display.
     *
     * @return a String value containing the formatted Appointment Date
     */
    public String getAppointmentDate() {
        return DecoratorUtils.escapedValueOrUnknown(DecoratorUtils.
            formatDateTime(appointment.getAppointmentDateComposite()));
    }

    /*
     * This method retrieves the clinic name from the Appointment
     *
     * @return a String containing the Clinic name
     */
    public String getClinicName() {
        return DecoratorUtils.escapedValueOrBlank(appointment.getClinicName());
    }

    /*
     * This method retrieves the clinic phone from the Appointment
     *
     * @return a String containing the Clinic phone
     */
    public String getClinicPhone() {
        return DecoratorUtils.escapedValueOrBlank(appointment.getClinicPhone());
    }

    /*
     * This method retrieves the location of the Appointment
     *
     * @return a String containing the Facility name
     */
    public String getLocation() {
        return DecoratorUtils.escapedValueOrBlank(appointment.getLocation());
    }
    /*
     * This method converts the VistA appointment status value into the
     * corresponding MHV appointment status for display in the apppointment
     * details.
     *
     * @return A String containing the MHV status for the appointment.
     */
    public String getAppointmentStatus()
    {
    	//return AppointmentDecoratorUtil.getAppointmentStatus(appointment);
		String appointmentStatus = AppointmentDecoratorUtil.getAppointmentStatus(appointment);

		if(appointmentStatus.equalsIgnoreCase(CANCELLED)){
			// <span style="color:#ff0000">Text you want to be red</span>
			//String span1 = "&lt;span style=&quotcolor:#ff0000&quot &gt;";
			//String span2 = "&lt;/span&gt;";

			String span1 = "<span style=\"color:#ff0000\">";
			String span2 = "</span>";
			appointmentStatus = span1 + appointmentStatus + span2;
			return appointmentStatus;
		}else
    	return appointmentStatus;
    }

    /*
     *  This method examines the VistA appointment type and sets the Compensation
     *  and Pension Appointment value accordingly.
     *
     *  @return A String value containing the value to display in the Compensation and Pension field
     */
    public String getCompensationAndPensionStatus()
    {
    	String compAppt = appointment.getAppointmentType();
    	if (compAppt != null)
    	{
    		return (compAppt.equalsIgnoreCase(CP)) ? escapedValue(COMPENSATION_APPT): "";
    	}
    	else
    		return "";
    }

    /*
     *  This method retrieve the Imprecise Lab Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted Lab Date and Time
     */
    public String getLabDateTime() {
        ImpreciseDateComposite labDate = appointment.getLabDateComposite();
        if (labDate == null ||
            	((labDate.getImpreciseValue() == null && labDate.getValue() == null) ||
                      	 (labDate.getImpreciseValue() == "" && labDate.getValue() == null))) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(labDate);
        }
    }

    /*
     *  This method retrieve the Imprecise EKG Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted EKG Date and Time
     */
    public String getEKGDateTime() {
        ImpreciseDateComposite ekgDate = appointment.getEkgDateComposite();
        if (ekgDate == null ||
            	((ekgDate.getImpreciseValue() == null && ekgDate.getValue() == null) ||
                   	 (ekgDate.getImpreciseValue() == "" && ekgDate.getValue() == null)))  {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(ekgDate);
        }
    }

    /*
     *  This method retrieve the Imprecise X-Ray Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted X-Ray Data and Time
     */
    public String getXrayDateTime() {
        ImpreciseDateComposite xrayDate = appointment.getXrayDateComposite();
        if (xrayDate == null ||
            	((xrayDate.getImpreciseValue() == null && xrayDate.getValue() == null) ||
            	 (xrayDate.getImpreciseValue() == "" && xrayDate.getValue() == null))) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(xrayDate);
        }
    }

}
